
/************************************************************************
 *
 * \file: DeviceMonitor.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Demo application
 *
 * \author: J. Harder / ADITG/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include "DeviceMonitor.h"

#include <sstream>
#include <adit_logging.h>

LOG_IMPORT_CONTEXT(tbdcl)

namespace adit { namespace bdcl {

using namespace uspi;

DeviceMonitor::DeviceMonitor()
{
    running = true;
    discoverer = nullptr;
    resetInfo = nullptr;
    pthread_mutex_init(&resetMutex, NULL);
}

DeviceMonitor::~DeviceMonitor()
{
    stopDeviceMonitor();
    pthread_mutex_destroy(&resetMutex);
}

bool DeviceMonitor::startDeviceMonitor(uint32_t inEventMask)
{
    running = true;

    discoverer = new FeatureDiscovery(this, inEventMask);

    /* Start Discoverer
     * Start monitoring of mobile device which supporting AOA protocl (AOAP) */
    if (DiscoveryError::OK != discoverer->start())
    {
        LOG_ERROR((tbdcl, "failed to start FeatureDiscovery"));
        discoverer = nullptr;
        return false;
    }
    else
    {
        LOG_INFO((tbdcl, "%s()  Use udevadm to trigger add event(s)", __func__));
        int32_t err = system("udevadm trigger --type=devices --subsystem-match=usb --action=add");
        if (err != 0)
        {
            LOG_ERROR((tbdcl, "%s()  Failed to trigger add event(s). system() returned with err = %d", __func__, err));
        }
    }

    return true;
}

void DeviceMonitor::stopDeviceMonitor()
{
    if (discoverer != nullptr)
    {
        if (DiscoveryError::OK != discoverer->stop())
        {
            LOG_ERROR((tbdcl, "failed to properly stop FeatureDiscovery"));
            // just continue though
        }
        delete discoverer;
        discoverer = nullptr;
    }
}

void DeviceMonitor::waitForExit()
{

    std::unique_lock<std::mutex> lckWaitForExit(mutBdclDemo);
    while (running)
    {
        cvBdclDemo.wait(lckWaitForExit);
    }
    if (discoverer != nullptr)
    {
        resetDevice();
    }else
    {
        LOG_ERROR((tbdcl, "DeviceMonitor::%s()  No discoverer", __func__));
    }
}

void DeviceMonitor::requestStop()
{
    // only trigger waitForExit to end
    std::unique_lock<std::mutex> lckWaitForExit(mutBdclDemo);
    running = false;
    cvBdclDemo.notify_one();
}

void DeviceMonitor::waitForDeviceReset(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    DiscoveryError res = DiscoveryError::INCOMPLETE;
    int32_t cnt = 0;
    DeviceInfo ddInfo = inUsbDevice->getInfo();

    /* resetDevice works only for AOAP devices. */
    if (DD_USB_AOAP == inUsbDevice->getEventMask()) {
        while ((res == DiscoveryError::INCOMPLETE) && (cnt < 8))
        {
            res = inUsbDevice->resetDevice(FD_PROTOCOL_GOOGLE_AOAP, 1000);
            if (DiscoveryError::INCOMPLETE == res) {
                LOG_INFO((tbdcl, "waitForDeviceReset() unable to reset device: %s, res = %s",
                         to_string(ddInfo).c_str(), to_str(res) ));
                sleep(1);
            } else if (DiscoveryError::OK != res) {
                LOG_WARN((tbdcl, "waitForDeviceReset() reset device: %s failed, res = %s",
                         to_string(ddInfo).c_str(), to_str(res) ));
                break;
            } else {
            }
            cnt++;
        }
    }
}
void DeviceMonitor::resetDevice()
{
    if (resetInfo != nullptr)
    {
        waitForDeviceReset(resetInfo);
    }
}

void DeviceMonitor::resetDevice(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    // TODO t_usbDeviceInformation should not be used as we do not know if copy is safe
    pthread_mutex_lock(&resetMutex);
    resetInfos.push_back(inUsbDevice);
    pthread_mutex_unlock(&resetMutex);
    resetFlag.clear();
}

DiscoveryError DeviceMonitor::foundCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (getRunningStatus()) {
        if (nullptr != inDevice) {

            onDeviceFound(inDevice);

        } else {
            LOG_ERROR((tbdcl, "DeviceMonitor::%s()  Error: device object is invalid", __func__));
        }
    }

    return res;
}

DiscoveryError DeviceMonitor::lostCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (getRunningStatus()) {
        if (nullptr != inDevice) {

            onDeviceLost(inDevice);

        } else {
            LOG_INFO((tbdcl, "DeviceMonitor::%s()  Error: device info is null", __func__));
        }
    }

    return res;
}

DiscoveryError DeviceMonitor::switchedCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (nullptr != inDevice) {

        onDeviceSwitched(inDevice);
        resetInfo = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inDevice);

    } else {
        LOG_INFO((tbdcl, "DeviceMonitor::%s()  Error: device info is null", __func__));
    }

    return res;
}

DiscoveryError DeviceMonitor::changedCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (nullptr != inDevice) {

        onDeviceChanged(inDevice);

    } else {
        LOG_INFO((tbdcl, "DeviceMonitor::%s()  Error: device info is null", __func__));
    }

    return res;
}

DiscoveryError DeviceMonitor::errorCb(adit::uspi::DiscoveryError inErrorCode)
{
    LOG_INFO((tbdcl, "Error %s occurred", to_str(inErrorCode) ));

    //todo: implement onDeviceError;
    return DiscoveryError::OK;
}

std::string to_string(DeviceInfo& inVal)
{
    std::stringstream ss;
    ss << "vendorId=0x" << std::hex << inVal.getiDeviceInfo(DSYS_IDVENDOR) <<
          " productId=0x" << std::hex << inVal.getiDeviceInfo(DSYS_IDPRODUCT) <<
          " serial=" << inVal.getDeviceInfo(DSYS_SERIAL);
    return ss.str();
}

} } // namespace adit { namespace bdcl {
